// TreeCtrlEx.cpp : implementation
//////////////////////////////////////////////////////////////////////

//some of this code is from www.codeguru.com

#include "stdafx.h"
#include "TreeCtrlEx.h"
#include "resourcehelper.h"   // temporary resource class
#include "resource.h"

#include "DockPane.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CTreeCtrlEx

CTreeCtrlEx::CTreeCtrlEx() : DRAG_DELAY(80) {
	m_bLDragging = FALSE;
	m_bDragEnabled = FALSE;
	m_nMenuID = -1;
	m_bParentDrag = FALSE;

}

CTreeCtrlEx::~CTreeCtrlEx()
{
}

BEGIN_MESSAGE_MAP(CTreeCtrlEx, CTreeCtrl)
	ON_WM_CONTEXTMENU()
	//{{AFX_MSG_MAP(CTreeCtrlEx)
	ON_NOTIFY_REFLECT(TVN_BEGINDRAG, OnBegindrag)
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONUP()
	ON_WM_LBUTTONDOWN()
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTreeCtrlEx message handlers

// nID		- resource id of image list bitmap
// nSize	- size in pixels of each image's width
// crMask	- bitmap transparent color

void CTreeCtrlEx::CreateImageList(int nID, int nSize, COLORREF crMask ) {
	m_image.Create( nID, nSize, 1, crMask );
	SetImageList( &m_image, TVSIL_NORMAL );
}

// dwStyle - style of tree can be one of the following: TVS_HASBUTTONS,
//			 TVS_EDITLABELS, TVS_HASLINES, TVS_HASLINESATROOT

void CTreeCtrlEx::SetTreeStyle(DWORD dwStyle) {
	ModifyStyle( 0, dwStyle );
}

// bDrag - TRUE enables drag operations for tree control

void CTreeCtrlEx::EnableDragOperations( BOOL bDrag ) {
	m_bDragEnabled = bDrag;
}

// uID - resource id of the popup menu for tree control

void CTreeCtrlEx::SetPopupMenuID(UINT uID) {
	m_nMenuID = uID;
}

// bEnable - TRUE enables parent items to be dragged

void CTreeCtrlEx::DisableParentDragging(BOOL bEnable) {
	m_bParentDrag = bEnable;
}

//set the parent of the tree
void CTreeCtrlEx::SetParent(CDockPane* pParent)
{
	m_pParentWnd = pParent;
}

// hParent		- parent item or NULL
// strText		- text string for tree item
// hInsAfter	- can be one of the following: TVI_ROOT, TVI_FIRST, TVI_LAST
// iImage		- index into image list
// bChildren	- TRUE item has children
//
// returns the handle of the newly created tree item

HTREEITEM CTreeCtrlEx::AddTreeItem( HTREEITEM hParent, CString strText,
								    HTREEITEM hInsAfter, int iImage, BOOL bChildren)
{
	HTREEITEM hItem;       // return value
	TV_ITEM tvI;           // item structure
	TV_INSERTSTRUCT tvIns; // item insert structure

	DWORD dwStyle = TVIF_TEXT | TVIF_IMAGE | TVIF_SELECTEDIMAGE | TVIF_PARAM;
	
	// Item has children
	if( bChildren ) dwStyle |= TVIF_CHILDREN;

	tvI.mask		= dwStyle;
	tvI.pszText		= strText.GetBuffer(1);
	tvI.cchTextMax	= strText.GetLength();
	tvI.iImage		= iImage;

	// Get type of icon to display
	if( bChildren ) {

		// Does not have a parent, OK to change bitmap
		if( hParent != NULL )	{
			tvI.iSelectedImage = iImage+1;
		}

		else {
			// Selected the base icon, don't change bitmap!!
			tvI.iSelectedImage = iImage;
		}
	}

	else {
		tvI.iSelectedImage = iImage;
	}

	// allow one more level down
	tvI.cChildren = 1;
	
	// do the insert
	tvIns.item			= tvI;
	tvIns.hInsertAfter	= hInsAfter;
	tvIns.hParent		= hParent;
	hItem				= InsertItem(&tvIns);

	return( hItem ); // return (add error trap here)
}

// hTreeItem - branch to expand

void CTreeCtrlEx::ExpandBranch(HTREEITEM hTreeItem)
{
	if( ItemHasChildren( hTreeItem )) {
		Expand( hTreeItem, TVE_EXPAND );
		hTreeItem = GetChildItem( hTreeItem );

		do {
			ExpandBranch( hTreeItem );
		} while (( hTreeItem = GetNextSiblingItem( hTreeItem )) != NULL );
	}

	EnsureVisible( GetSelectedItem());
}

// hTreeItem - branch to collapse

void CTreeCtrlEx::CollapseBranch(HTREEITEM hTreeItem)
{
	if( ItemHasChildren( hTreeItem )) {
		Expand( hTreeItem, TVE_COLLAPSE );
		hTreeItem = GetChildItem( hTreeItem );

		do {
			CollapseBranch( hTreeItem );
		} while (( hTreeItem = GetNextSiblingItem( hTreeItem )) != NULL );
	}
}

void CTreeCtrlEx::CollapseAll()
{
	HTREEITEM hTreeItem = GetRootItem();
	do {
		CollapseBranch( hTreeItem );
	} while (( hTreeItem = GetNextSiblingItem( hTreeItem )) != NULL );
}

// Moving an item or a branch is a combination of copying followed by deleting.
// To move an item use CopyItem() followed by DeleteItem().
// To move a branch use CopyBranch() followed by DeleteItem().
// Note that when you delete an item then any child item is also deleted.

// CopyItem - Copies an item to a new location
// Returns - Handle of the new item
// hItem - Item to be copied
// htiNewParent - Handle of the parent for new item
// htiAfter - Item after which the new item should be created

HTREEITEM CTreeCtrlEx::CopyItem(HTREEITEM hItem, HTREEITEM htiNewParent, HTREEITEM htiAfter)
{
	TV_INSERTSTRUCT	tvstruct;
	HTREEITEM		hNewItem;
	CString			sText;

	// get information of the source item
	tvstruct.item.hItem = hItem;
	tvstruct.item.mask	= TVIF_CHILDREN | TVIF_HANDLE | TVIF_IMAGE | TVIF_SELECTEDIMAGE;

	GetItem( &tvstruct.item );
	sText = GetItemText( hItem );

	tvstruct.item.cchTextMax	= sText.GetLength();
	tvstruct.item.pszText		= sText.LockBuffer();

	// Insert the item at proper location
	tvstruct.hParent			= htiNewParent;
	tvstruct.hInsertAfter		= htiAfter;
	tvstruct.item.mask			= TVIF_IMAGE | TVIF_SELECTEDIMAGE | TVIF_TEXT;
	hNewItem					= InsertItem( &tvstruct );
	sText.ReleaseBuffer();

	// Now copy item data and item state.
	SetItemData( hNewItem, GetItemData( hItem ));
	SetItemState( hNewItem, GetItemState( hItem, TVIS_STATEIMAGEMASK ),
		TVIS_STATEIMAGEMASK );

	// Call virtual function to allow further processing in derived class
	OnItemCopied( hItem, hNewItem );
	return hNewItem;
}

void CTreeCtrlEx::OnItemCopied(HTREEITEM hItem, HTREEITEM hNewItem)
{

}

// CopyBranch- Copies all items in a branch to a new location
// Returns- The new branch node
// htiBranch- The node that starts the branch
// htiNewParent - Handle of the parent for new branch
// htiAfter- Item after which the new branch should be created

HTREEITEM CTreeCtrlEx::CopyBranch(HTREEITEM htiBranch, HTREEITEM htiNewParent, HTREEITEM htiAfter)
{
	HTREEITEM hChild;
	HTREEITEM hNewItem  = CopyItem( htiBranch, htiNewParent, htiAfter );
	hChild = GetChildItem( htiBranch );

	while( hChild != NULL ) {
		// Recursively transfer all the items
		CopyBranch( hChild, hNewItem );
		hChild = GetNextSiblingItem( hChild );
	}

	return hNewItem;
}

void CTreeCtrlEx::OnBegindrag(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NM_TREEVIEW* pNMTreeView = (NM_TREEVIEW*)pNMHDR;
	*pResult = 0;
	
	// This code is to prevent accidental drags.
	if((int)( GetTickCount() - m_dwDragStart ) < DRAG_DELAY ) {
		return;
	}

	// if drag is not enabled, return
	if( !m_bDragEnabled ) {
		return;
	}

	// check to make sure that we are not dragging a
	// parent item.
	if( ItemHasChildren( pNMTreeView->itemNew.hItem ) && ( !m_bParentDrag )) {
		return;
	}

	// Set up the timer
	m_nTimerID = SetTimer(1, 75, NULL );

	m_hitemDrag = pNMTreeView->itemNew.hItem;
	m_hitemDrop = NULL;
	
	// get the image list for dragging
	m_pDragImage = CreateDragImage( m_hitemDrag );
	
	// CreateDragImage() returns NULL if no image list
	// associated with the tree view control
	if( !m_pDragImage )
		return;
	
	m_bLDragging = TRUE;
	m_pDragImage->BeginDrag(0, CPoint(-15,-15));
	POINT pt = pNMTreeView->ptDrag;
	ClientToScreen( &pt );
	m_pDragImage->DragEnter(NULL, pt);
	SetCapture();
	
	*pResult = 0;
}

void CTreeCtrlEx::OnMouseMove(UINT nFlags, CPoint point) 
{
	HTREEITEM	hitem;
	UINT		flags;
	
	if( m_bLDragging )
	{
		POINT pt = point;
		ClientToScreen( &pt );
		CImageList::DragMove( pt );
		
		if (( hitem = HitTest( point, &flags )) != NULL )
		{
			CImageList::DragShowNolock( FALSE );
			SelectDropTarget(hitem);
			m_hitemDrop = hitem;
			CImageList::DragShowNolock( TRUE );
		}
	}
	
	CTreeCtrl::OnMouseMove(nFlags, point);
}

void CTreeCtrlEx::OnLButtonUp(UINT nFlags, CPoint point) 
{
	if (m_bLDragging)
	{
		m_bLDragging = FALSE;
		CImageList::DragLeave(this);
		CImageList::EndDrag();
		ReleaseCapture();
		
		delete m_pDragImage;
		
		// Remove drop target highlighting
		SelectDropTarget(NULL);
		
		if( m_hitemDrag == m_hitemDrop )
			return;

		// If Drag item is an ancestor of Drop item then return
		HTREEITEM htiParent = m_hitemDrop;
		while( (htiParent = GetParentItem( htiParent )) != NULL )
		{
			if( htiParent == m_hitemDrag ) return;
		}
		
		Expand( m_hitemDrop, TVE_EXPAND ) ;
		
		HTREEITEM htiNew = CopyBranch( m_hitemDrag, m_hitemDrop, TVI_LAST );
		DeleteItem(m_hitemDrag);
		SelectItem( htiNew );
		KillTimer( m_nTimerID );
	}
	
	CTreeCtrl::OnLButtonUp(nFlags, point);
}

void CTreeCtrlEx::OnLButtonDown(UINT nFlags, CPoint point) 
{
	m_dwDragStart = GetTickCount();	
	CTreeCtrl::OnLButtonDown(nFlags, point);
}

BOOL CTreeCtrlEx::PreTranslateMessage(MSG* pMsg) 
{
	// Cancel dragging if escape key pressed.
	if( pMsg->message == WM_KEYDOWN && pMsg->wParam == VK_ESCAPE && m_bLDragging )
	{
		m_bLDragging = FALSE;
		CImageList::EndDrag();
		ReleaseCapture();
		SelectDropTarget(NULL);
		KillTimer( m_nTimerID );
		delete m_pDragImage;
		return TRUE;
	}
		
	return CTreeCtrl::PreTranslateMessage(pMsg);
}

void CTreeCtrlEx::OnTimer(UINT nIDEvent) 
{
	if( nIDEvent != m_nTimerID )
	{
		CTreeCtrl::OnTimer(nIDEvent);
		return;
	}
	
	// Doesn't matter that we didn't initialize m_timerticks
	m_timerticks++;
	
	POINT pt;
	GetCursorPos( &pt );
	RECT rect;
	GetClientRect( &rect );
	ClientToScreen( &rect );
	
	// NOTE: Screen coordinate is being used because the call
	// to DragEnter had used the Desktop window.
	CImageList::DragMove(pt);
	
	HTREEITEM hitem = GetFirstVisibleItem();
	
	if( pt.y < rect.top + 10 )
	{
		// We need to scroll up
		// Scroll slowly if cursor near the treeview control
		int slowscroll = 6 - (rect.top + 10 - pt.y) / 20;
		if( 0 == ( m_timerticks % (slowscroll > 0? slowscroll : 1) ) )
		{
			CImageList::DragShowNolock(FALSE);
			SendMessage( WM_VSCROLL, SB_LINEUP);
			SelectDropTarget(hitem);
			m_hitemDrop = hitem;
			CImageList::DragShowNolock(TRUE);
		}
	}
	else if( pt.y > rect.bottom - 10 )
	{
		// We need to scroll down
		// Scroll slowly if cursor near the treeview control
		int slowscroll = 6 - (pt.y - rect.bottom + 10 ) / 20;
		if( 0 == ( m_timerticks % (slowscroll > 0? slowscroll : 1) ) )
		{
			CImageList::DragShowNolock(FALSE);
			SendMessage( WM_VSCROLL, SB_LINEDOWN);
			int nCount = GetVisibleCount();
			for ( int i=0; i<nCount-1; ++i )
				hitem = GetNextVisibleItem(hitem);
			if( hitem )
				SelectDropTarget(hitem);
			m_hitemDrop = hitem;
			CImageList::DragShowNolock(TRUE);
		}
	}
	
	CTreeCtrl::OnTimer(nIDEvent);
}

// nIcon - index into image list

void CTreeCtrlEx::SetSelectedIcon(int nIcon)
{
	HTREEITEM hItem = GetSelectedItem();
	SetItemImage( hItem, nIcon, nIcon );
}

UINT CTreeCtrlEx::GetSelectedIcon()
{
	HTREEITEM hItem = GetSelectedItem();
	int nImage;
	int nSelectedImage;

	if( !GetItemImage( hItem, nImage, nSelectedImage )) {
		nImage = -1;
	}

	return nImage;
}

void CTreeCtrlEx::OnContextMenu(CWnd*, CPoint point)
{
	// if no popup menu was defined, just return.
	if( m_nMenuID == -1 ) {
		return;
	}

	// CG: This block was added by the Pop-up Menu component
	{
		if (point.x == -1 && point.y == -1){
			//keystroke invocation
			CRect rect;
			GetClientRect(rect);
			ClientToScreen(rect);

			point = rect.TopLeft();
			point.Offset(5, 5);
		}

		CMenu menu;
		VERIFY(menu.LoadMenu( m_nMenuID ));

		CMenu* pPopup = menu.GetSubMenu(0);
		ASSERT(pPopup != NULL);
		if (pPopup == NULL)
		{
			return;
		}
		CWnd* pWndPopupOwner = this;

		while (pWndPopupOwner->GetStyle() & WS_CHILD)
			pWndPopupOwner = pWndPopupOwner->GetParent();

		pPopup->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, point.x, point.y,
			pWndPopupOwner);
	}
}

// GetLastItem - Gets last item in the branch
// Returns - Last item
// hItem - Node identifying the branch. NULL will
// return the last item in outine

HTREEITEM CTreeCtrlEx::GetLastItem(HTREEITEM hItem)
{
	// Last child of the last child of the last child ...
	HTREEITEM htiNext;
	
	if( hItem == NULL ){
		// Get the last item at the top level
		htiNext = GetRootItem();
		while( htiNext ){
			hItem = htiNext;
			htiNext = GetNextSiblingItem( htiNext );
		}
	}
	
	while( ItemHasChildren( hItem ) ){
		htiNext = GetChildItem( hItem );
		
		while( htiNext ){
			hItem = htiNext;
			htiNext = GetNextSiblingItem( htiNext );
		}
	}
	
	return hItem;
}

// GetNextItem - Get next item as if outline was completely expanded
// Returns - The item immediately below the reference item
// hItem - The reference item

HTREEITEM CTreeCtrlEx::GetNextItem(HTREEITEM hItem)
{
	HTREEITEM hti;

	if( ItemHasChildren( hItem ))
		return GetChildItem( hItem );	// return first child

	else {
		// return next sibling item
		// go up the tree to find a parent's sibling if needed.
		while(( hti = GetNextSiblingItem( hItem )) == NULL ) {
			if(( hItem = GetParentItem( hItem )) == NULL ) {
				return NULL;
			}
		}
	}

	return hti;
}

// GetNextItem - Get previous item as if outline was completely expanded
// Returns- The item immediately above the reference item
// hItem- The reference item

HTREEITEM CTreeCtrlEx::GetPrevItem(HTREEITEM hItem)
{
	HTREEITEM hti;

	hti = GetPrevSiblingItem( hItem );

	if( hti == NULL ) {
		hti = GetParentItem( hItem );
	}
	else {
		hti = GetLastItem( hti );
	}

	return hti;
}

// FindItem			- Finds an item that contains the search string
// Returns			- Handle to the item or NULL
// str				- String to search for
// bCaseSensitive	- Should the search be case sensitive
// bDownDir			- Search direction - TRUE for down
// bWholeWord		- True if search should match whole words
// hItem			- Item to start searching from. NULL for
//					- currently selected item

HTREEITEM CTreeCtrlEx::FindItem(CString & str, BOOL bCaseSensitive, BOOL bDownDir, BOOL bWholeWord, HTREEITEM hItem)
{
	int lenSearchStr = str.GetLength();
	if( lenSearchStr == 0 ) return NULL;
	
	HTREEITEM htiSel = hItem ? hItem : GetSelectedItem();
	HTREEITEM htiCur = bDownDir ? GetNextItem( htiSel ) : GetPrevItem( htiSel );
	CString sSearch = str;
	
	if( htiCur == NULL )
	{
		if( bDownDir ) htiCur = GetRootItem();
		else htiCur = GetLastItem( NULL );
	}
	
	if( !bCaseSensitive )
		sSearch.MakeLower();
	
	while( htiCur && htiCur != htiSel )
	{
		CString sItemText = GetItemText( htiCur );
		if( !bCaseSensitive )
			sItemText.MakeLower();
		
		int n;
		while( (n = sItemText.Find( sSearch )) != -1 )
		{
			// Search string found
			if( bWholeWord )
			{
				// Check preceding char
				if( n != 0 )
				{
					if( isalpha( sItemText[n-1] ) ||
						sItemText[n-1] == '_' ) {

						// Not whole word
						sItemText = sItemText.Right(
							sItemText.GetLength() - n -
							lenSearchStr );
						continue;
					}
				}
				
				// Check succeeding char
				if( sItemText.GetLength() > n + lenSearchStr
					&& ( isalpha(sItemText[n+lenSearchStr]) ||
					sItemText[n+lenSearchStr] == '_' ) )
				{
					// Not whole word
					sItemText = sItemText.Right( sItemText.GetLength()
						- n - sSearch.GetLength());
					continue;
				}
			}
			
			if( IsFindValid( htiCur ) )
				return htiCur;
			else break;
		}
		
		
		htiCur = bDownDir ? GetNextItem( htiCur ) : GetPrevItem( htiCur );
		if( htiCur == NULL )
		{
			if( bDownDir ) htiCur = GetRootItem();
			else htiCur =  GetLastItem( NULL );
		}
	}
	return NULL;
}

// IsFindValid	- Virtual function used by FindItem to allow this
//				  function to filter the result of FindItem
// Returns		- True if item matches the criteria
// Arg			- Handle of the item

BOOL CTreeCtrlEx::IsFindValid(HTREEITEM) {
	return TRUE;
}

